package eu.isas.peptideshaker.gui.export;

import com.compomics.util.experiment.biology.modifications.ModificationFactory;
import com.compomics.util.gui.file_handling.FileAndFileFilter;
import com.compomics.util.experiment.identification.spectrum_annotation.AnnotationParameters;
import com.compomics.util.experiment.io.biology.protein.FastaSummary;
import com.compomics.util.experiment.io.identification.MzIdentMLVersion;
import com.compomics.util.experiment.io.parameters.SdrfExport;
import com.compomics.util.gui.JOptionEditorPane;
import com.compomics.util.gui.error_handlers.HelpDialog;
import com.compomics.util.gui.file_handling.FileChooserUtil;
import com.compomics.util.gui.waiting.waitinghandlers.ProgressDialogX;
import com.compomics.util.io.IoUtil;
import com.compomics.util.io.export.ExportWriter;
import com.compomics.util.io.file.LastSelectedFolder;
import com.compomics.util.parameters.identification.IdentificationParameters;
import eu.isas.peptideshaker.PeptideShaker;
import eu.isas.peptideshaker.export.MzIdentMLExport;
import eu.isas.peptideshaker.gui.PeptideShakerGUI;
import eu.isas.peptideshaker.preferences.ProjectDetails;
import java.awt.Color;
import java.awt.Toolkit;
import java.io.*;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import javax.swing.*;
import org.xml.sax.SAXException;
import uk.ac.ebi.pride.tools.ErrorHandlerIface;
import uk.ac.ebi.pride.tools.GenericSchemaValidator;
import uk.ac.ebi.pride.tools.ValidationErrorHandler;

/**
 * A dialog where the user can export the project to mzIdentML.
 *
 * @author Harald Barsnes
 */
public class MzIdentMLExportDialog extends javax.swing.JDialog {

    /**
     * A simple progress dialog.
     */
    private static ProgressDialogX progressDialog;
    /**
     * The PeptideShakerGUI main class.
     */
    private final PeptideShakerGUI peptideShakerGUI;
    /**
     * If true, the created mzid file will be validated against the mzid 1.1
     * schema.
     */
    private final boolean validateMzIdentML = false; // just takes too long if switched on... (note: only works if the mzid file is not zipped)
    /**
     * The version of mzIdentML to use.
     */
    private MzIdentMLVersion mzIdentMLVersion = MzIdentMLVersion.v1_1;
    /**
     * Enables the export of an sdrf file next to the mzIdentML file.
     */
    private boolean exportSdrf = true;

    /**
     * Create a new MzIdentMLExportDialog.
     *
     * @param peptideShakerGUI a reference to the main GUI frame
     * @param modal if the dialog is to be modal or not
     */
    public MzIdentMLExportDialog(PeptideShakerGUI peptideShakerGUI, boolean modal) {

        super(peptideShakerGUI, modal);
        this.peptideShakerGUI = peptideShakerGUI;

        initComponents();

        // set gui properties
        setGuiProperties();

        // insert project data
        insertProjectData();

        // validate the input
        validateInput();

        setLocationRelativeTo(peptideShakerGUI);
        setVisible(true);
    }

    /**
     * Set the GUI properties.
     */
    private void setGuiProperties() {

    }

    /**
     * Insert the available project data.
     */
    private void insertProjectData() {

        ProjectDetails projectDetails = peptideShakerGUI.getProjectDetails();

        // use the saved mzIdentML annotation, if any
        contactFirstNameJTextField.setText(projectDetails.getContactFirstName());
        contactLastNameJTextField.setText(projectDetails.getContactLastName());
        contactEmailJTextField.setText(projectDetails.getContactEmail());
        contactAddressJTextField.setText(projectDetails.getContactAddress());
        contactUrlJTextField.setText(projectDetails.getContactUrl());

        organizationNameJTextField.setText(projectDetails.getOrganizationName());
        organizationEmailJTextField.setText(projectDetails.getOrganizationEmail());
        organizationAddressJTextField.setText(projectDetails.getOrganizationAddress());
        organizationUrlJTextField.setText(projectDetails.getOrganizationUrl());

        includeSequencesCheckBox.setSelected(projectDetails.getIncludeProteinSequences());

        if (projectDetails.getMzIdentMLOutputFile() != null
                && new File(projectDetails.getMzIdentMLOutputFile()).exists()) {
            outputFolderJTextField.setText(projectDetails.getMzIdentMLOutputFile());
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        backgroundJPanel = new javax.swing.JPanel();
        contactPanel = new javax.swing.JPanel();
        contactFirstNameLabel = new javax.swing.JLabel();
        contactFirstNameJTextField = new javax.swing.JTextField();
        contactLastNameLabel = new javax.swing.JLabel();
        contactLastNameJTextField = new javax.swing.JTextField();
        contactEmailLabel = new javax.swing.JLabel();
        contactEmailJTextField = new javax.swing.JTextField();
        contactAddressLabel = new javax.swing.JLabel();
        contactAddressJTextField = new javax.swing.JTextField();
        contactUrlLabel = new javax.swing.JLabel();
        contactUrlJTextField = new javax.swing.JTextField();
        organizationPanel = new javax.swing.JPanel();
        organizationNameLabel = new javax.swing.JLabel();
        organizationNameJTextField = new javax.swing.JTextField();
        organizationEmailLabel = new javax.swing.JLabel();
        organizationEmailJTextField = new javax.swing.JTextField();
        organizationAddressLabel = new javax.swing.JLabel();
        organizationAddressJTextField = new javax.swing.JTextField();
        organizationUrlLabel = new javax.swing.JLabel();
        organizationUrlJTextField = new javax.swing.JTextField();
        outputPanel = new javax.swing.JPanel();
        outputFolderLabel = new javax.swing.JLabel();
        outputFolderJTextField = new javax.swing.JTextField();
        browseOutputFolderJButton = new javax.swing.JButton();
        includeSequencesCheckBox = new javax.swing.JCheckBox();
        helpLabel = new javax.swing.JLabel();
        openDialogHelpJButton = new javax.swing.JButton();
        convertJButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Export mzIdentML");
        setResizable(false);

        backgroundJPanel.setBackground(new java.awt.Color(230, 230, 230));

        contactPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Contact"));
        contactPanel.setOpaque(false);

        contactFirstNameLabel.setForeground(new java.awt.Color(255, 0, 0));
        contactFirstNameLabel.setText("First Name");

        contactFirstNameJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        contactFirstNameJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                contactFirstNameJTextFieldKeyReleased(evt);
            }
        });

        contactLastNameLabel.setForeground(new java.awt.Color(255, 0, 0));
        contactLastNameLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        contactLastNameLabel.setText("Last Name");

        contactLastNameJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        contactLastNameJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                contactLastNameJTextFieldKeyReleased(evt);
            }
        });

        contactEmailLabel.setForeground(new java.awt.Color(255, 0, 0));
        contactEmailLabel.setText("E-mail");

        contactEmailJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        contactEmailJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                contactEmailJTextFieldKeyReleased(evt);
            }
        });

        contactAddressLabel.setForeground(new java.awt.Color(255, 0, 0));
        contactAddressLabel.setText("Address");

        contactAddressJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        contactAddressJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                contactAddressJTextFieldKeyReleased(evt);
            }
        });

        contactUrlLabel.setText("URL");

        contactUrlJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        contactUrlJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                contactUrlJTextFieldKeyReleased(evt);
            }
        });

        javax.swing.GroupLayout contactPanelLayout = new javax.swing.GroupLayout(contactPanel);
        contactPanel.setLayout(contactPanelLayout);
        contactPanelLayout.setHorizontalGroup(
            contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(contactPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addGroup(contactPanelLayout.createSequentialGroup()
                        .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(contactFirstNameLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(contactEmailLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(contactAddressLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                            .addComponent(contactEmailJTextField, javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(javax.swing.GroupLayout.Alignment.LEADING, contactPanelLayout.createSequentialGroup()
                                .addComponent(contactFirstNameJTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 276, Short.MAX_VALUE)
                                .addGap(18, 18, 18)
                                .addComponent(contactLastNameLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 39, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(contactLastNameJTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 171, Short.MAX_VALUE))
                            .addComponent(contactAddressJTextField)))
                    .addGroup(contactPanelLayout.createSequentialGroup()
                        .addComponent(contactUrlLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(contactUrlJTextField)))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        contactPanelLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {contactFirstNameJTextField, contactLastNameJTextField});

        contactPanelLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {contactFirstNameLabel, contactLastNameLabel});

        contactPanelLayout.setVerticalGroup(
            contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(contactPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(contactFirstNameLabel)
                    .addComponent(contactFirstNameJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(contactLastNameLabel)
                    .addComponent(contactLastNameJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(contactEmailLabel)
                    .addComponent(contactEmailJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(contactAddressLabel)
                    .addComponent(contactAddressJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(contactPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(contactUrlLabel)
                    .addComponent(contactUrlJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        organizationPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Organization"));
        organizationPanel.setOpaque(false);

        organizationNameLabel.setForeground(new java.awt.Color(255, 0, 0));
        organizationNameLabel.setText("Name");

        organizationNameJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        organizationNameJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                organizationNameJTextFieldKeyReleased(evt);
            }
        });

        organizationEmailLabel.setForeground(new java.awt.Color(255, 0, 0));
        organizationEmailLabel.setText("E-mail");

        organizationEmailJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        organizationEmailJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                organizationEmailJTextFieldKeyReleased(evt);
            }
        });

        organizationAddressLabel.setForeground(new java.awt.Color(255, 0, 0));
        organizationAddressLabel.setText("Address");

        organizationAddressJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        organizationAddressJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                organizationAddressJTextFieldKeyReleased(evt);
            }
        });

        organizationUrlLabel.setText("URL");

        organizationUrlJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));
        organizationUrlJTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                organizationUrlJTextFieldKeyReleased(evt);
            }
        });

        javax.swing.GroupLayout organizationPanelLayout = new javax.swing.GroupLayout(organizationPanel);
        organizationPanel.setLayout(organizationPanelLayout);
        organizationPanelLayout.setHorizontalGroup(
            organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(organizationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(organizationPanelLayout.createSequentialGroup()
                        .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(organizationNameLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(organizationEmailLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(organizationAddressLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(organizationEmailJTextField)
                            .addComponent(organizationNameJTextField)
                            .addComponent(organizationAddressJTextField)))
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, organizationPanelLayout.createSequentialGroup()
                        .addComponent(organizationUrlLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(organizationUrlJTextField)))
                .addContainerGap())
        );
        organizationPanelLayout.setVerticalGroup(
            organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(organizationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(organizationNameLabel)
                    .addComponent(organizationNameJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(organizationEmailLabel)
                    .addComponent(organizationEmailJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(organizationAddressLabel)
                    .addComponent(organizationAddressJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(organizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(organizationUrlLabel)
                    .addComponent(organizationUrlJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        outputPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Output File"));
        outputPanel.setOpaque(false);

        outputFolderLabel.setForeground(new java.awt.Color(255, 0, 0));
        outputFolderLabel.setText("File");
        outputFolderLabel.setToolTipText("The file where the mzIdentML export will be saved");

        outputFolderJTextField.setEditable(false);
        outputFolderJTextField.setToolTipText("The folder where the mzIdentML file will be saved");
        outputFolderJTextField.setMargin(new java.awt.Insets(2, 4, 2, 2));

        browseOutputFolderJButton.setText("Browse");
        browseOutputFolderJButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                browseOutputFolderJButtonActionPerformed(evt);
            }
        });

        includeSequencesCheckBox.setText("Include Protein Sequences");
        includeSequencesCheckBox.setToolTipText("Select to include the protein sequences in the mzIdentML file");
        includeSequencesCheckBox.setIconTextGap(10);

        javax.swing.GroupLayout outputPanelLayout = new javax.swing.GroupLayout(outputPanel);
        outputPanel.setLayout(outputPanelLayout);
        outputPanelLayout.setHorizontalGroup(
            outputPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, outputPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(outputFolderLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(outputFolderJTextField)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(browseOutputFolderJButton, javax.swing.GroupLayout.PREFERRED_SIZE, 80, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, outputPanelLayout.createSequentialGroup()
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(includeSequencesCheckBox)
                .addGap(106, 106, 106))
        );
        outputPanelLayout.setVerticalGroup(
            outputPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(outputPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(outputPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(outputFolderLabel)
                    .addComponent(outputFolderJTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(browseOutputFolderJButton))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(includeSequencesCheckBox)
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        helpLabel.setFont(helpLabel.getFont().deriveFont((helpLabel.getFont().getStyle() | java.awt.Font.ITALIC)));
        helpLabel.setText("Insert the required information and click Convert to export the project to mzIdentML.");

        openDialogHelpJButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/icons/help.GIF"))); // NOI18N
        openDialogHelpJButton.setToolTipText("Help");
        openDialogHelpJButton.setBorder(null);
        openDialogHelpJButton.setBorderPainted(false);
        openDialogHelpJButton.setContentAreaFilled(false);
        openDialogHelpJButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseEntered(java.awt.event.MouseEvent evt) {
                openDialogHelpJButtonMouseEntered(evt);
            }
            public void mouseExited(java.awt.event.MouseEvent evt) {
                openDialogHelpJButtonMouseExited(evt);
            }
        });
        openDialogHelpJButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                openDialogHelpJButtonActionPerformed(evt);
            }
        });

        convertJButton.setBackground(new java.awt.Color(0, 153, 0));
        convertJButton.setFont(convertJButton.getFont().deriveFont(convertJButton.getFont().getStyle() | java.awt.Font.BOLD));
        convertJButton.setForeground(new java.awt.Color(255, 255, 255));
        convertJButton.setText("Convert!");
        convertJButton.setToolTipText("Click here to start the conversion!");
        convertJButton.setEnabled(false);
        convertJButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                convertJButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout backgroundJPanelLayout = new javax.swing.GroupLayout(backgroundJPanel);
        backgroundJPanel.setLayout(backgroundJPanelLayout);
        backgroundJPanelLayout.setHorizontalGroup(
            backgroundJPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundJPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(backgroundJPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(backgroundJPanelLayout.createSequentialGroup()
                        .addGap(10, 10, 10)
                        .addComponent(openDialogHelpJButton, javax.swing.GroupLayout.PREFERRED_SIZE, 23, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(helpLabel)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(convertJButton, javax.swing.GroupLayout.PREFERRED_SIZE, 154, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(15, 15, 15))
                    .addComponent(outputPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(backgroundJPanelLayout.createSequentialGroup()
                        .addGroup(backgroundJPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                            .addComponent(organizationPanel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                            .addComponent(contactPanel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                        .addGap(0, 0, Short.MAX_VALUE)))
                .addContainerGap())
        );
        backgroundJPanelLayout.setVerticalGroup(
            backgroundJPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundJPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(contactPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(organizationPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(outputPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(backgroundJPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.CENTER)
                    .addComponent(openDialogHelpJButton)
                    .addComponent(helpLabel)
                    .addComponent(convertJButton, javax.swing.GroupLayout.PREFERRED_SIZE, 53, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundJPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundJPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Change the cursor to a hand cursor.
     *
     * @param evt
     */
    private void openDialogHelpJButtonMouseEntered(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_openDialogHelpJButtonMouseEntered
        setCursor(new java.awt.Cursor(java.awt.Cursor.HAND_CURSOR));
    }//GEN-LAST:event_openDialogHelpJButtonMouseEntered

    /**
     * Change the cursor back to the default cursor.
     *
     * @param evt
     */
    private void openDialogHelpJButtonMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_openDialogHelpJButtonMouseExited
        setCursor(new java.awt.Cursor(java.awt.Cursor.DEFAULT_CURSOR));
    }//GEN-LAST:event_openDialogHelpJButtonMouseExited

    /**
     * Open the help dialog.
     *
     * @param evt
     */
    private void openDialogHelpJButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_openDialogHelpJButtonActionPerformed
        
        setCursor(new java.awt.Cursor(java.awt.Cursor.WAIT_CURSOR));
        
        new HelpDialog(
                peptideShakerGUI, 
                getClass().getResource("/helpFiles/mzIdentMLExportDialog.html"),
                Toolkit.getDefaultToolkit().getImage(getClass().getResource("/icons/help.GIF")),
                Toolkit.getDefaultToolkit().getImage(getClass().getResource("/icons/peptide-shaker.gif")),
                "Export mzIdentML - Help"
        );
        setCursor(new java.awt.Cursor(java.awt.Cursor.DEFAULT_CURSOR));
        
    }//GEN-LAST:event_openDialogHelpJButtonActionPerformed

    /**
     * Opens a file chooser where the user can select the output file.
     *
     * @param evt
     */
    private void browseOutputFolderJButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_browseOutputFolderJButtonActionPerformed
        
        this.setCursor(new java.awt.Cursor(java.awt.Cursor.WAIT_CURSOR));

        // First check whether a file has already been selected.
        // If so, start from that file's parent.
        LastSelectedFolder lastSelectedFolder = peptideShakerGUI.getLastSelectedFolder();
        String folder = lastSelectedFolder.getLastSelectedFolder(ExportWriter.LAST_FOLDER_KEY);
        if (folder == null) {
            folder = lastSelectedFolder.getLastSelectedFolder();
        }

        if (!outputFolderJTextField.getText().isEmpty()) {
            if (new File(outputFolderJTextField.getText()).exists()) {
                folder = outputFolderJTextField.getText();
            } else if (new File(outputFolderJTextField.getText()).getParentFile().exists()) {
                folder = new File(outputFolderJTextField.getText()).getParent();
            }
        }

        MzIdentMLVersion[] mzIdentMLVersions = MzIdentMLVersion.values();
        String[] versionsDescriptions = new String[mzIdentMLVersions.length];
        for (int i = 0; i < mzIdentMLVersions.length; i++) {
            MzIdentMLVersion tempVersion = mzIdentMLVersions[i];
            StringBuilder stringBuilder = new StringBuilder(23);
            stringBuilder.append("mzIdentML ").append(tempVersion.name).append(" (*.mzid.gzip)");
            versionsDescriptions[i] = stringBuilder.toString();
        }

        int defaultFilterIndex;
        switch (mzIdentMLVersion) {
            case v1_1:
                defaultFilterIndex = 0;
                break;
            case v1_2:
                defaultFilterIndex = 1;
                break;
            default:
                throw new UnsupportedOperationException("mzIdentML version " + mzIdentMLVersion.name + " not supported.");
        }

        FileAndFileFilter selectedFileAndFilter = FileChooserUtil.getUserSelectedFile(
                this, 
                new String[]{".mzid.gzip", ".mzid.gzip"},
                versionsDescriptions, 
                "Select Export File",
                folder, 
                peptideShakerGUI.getProjectParameters().getProjectUniqueName(), 
                false, 
                true, 
                false, 
                defaultFilterIndex
        );

        if (selectedFileAndFilter != null) {
            String path = selectedFileAndFilter.getFile().getAbsolutePath();

            if (!path.endsWith(".mzid.gzip")) {
                path += ".mzid.gzip";
            }

            int index = -1;
            String selectedVersionDescription = selectedFileAndFilter.getFileFilter().getDescription();
            for (int i = 0; i < versionsDescriptions.length; i++) {
                String versionDescription = versionsDescriptions[i];
                if (versionDescription.equals(selectedVersionDescription)) {
                    index = i;
                    break;
                }
            }

            mzIdentMLVersion = MzIdentMLVersion.getMzIdentMLVersion(index);

            lastSelectedFolder.setLastSelectedFolder(ExportWriter.LAST_FOLDER_KEY, folder);
            outputFolderJTextField.setText(path);
        }

        validateInput();

        this.setCursor(new java.awt.Cursor(java.awt.Cursor.DEFAULT_CURSOR));
    }//GEN-LAST:event_browseOutputFolderJButtonActionPerformed

    /**
     * @see #validateInput()
     */
    private void contactFirstNameJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_contactFirstNameJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_contactFirstNameJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void contactLastNameJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_contactLastNameJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_contactLastNameJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void contactEmailJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_contactEmailJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_contactEmailJTextFieldKeyReleased

    /**
     * Convert the project to an mzIdentML file.
     *
     * @param evt
     */
    private void convertJButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_convertJButtonActionPerformed

        final File finalOutputFile = new File(outputFolderJTextField.getText());

        progressDialog = new ProgressDialogX(peptideShakerGUI,
                Toolkit.getDefaultToolkit().getImage(getClass().getResource("/icons/peptide-shaker.gif")),
                Toolkit.getDefaultToolkit().getImage(getClass().getResource("/icons/peptide-shaker-orange.gif")),
                true);
        progressDialog.setPrimaryProgressCounterIndeterminate(true);

        progressDialog.setTitle("Exporting mzIdentML. Please Wait...");

        new Thread(new Runnable() {
            public void run() {
                try {
                    progressDialog.setVisible(true);
                } catch (IndexOutOfBoundsException e) {
                    // ignore
                }
            }
        }, "ProgressDialog").start();

        new Thread("ConvertThread") {
            @Override
            public void run() {

                IdentificationParameters identificationParameters = peptideShakerGUI.getIdentificationParameters();
                AnnotationParameters annotationParameters = identificationParameters.getAnnotationParameters();
                ProjectDetails projectDetails = peptideShakerGUI.getProjectDetails();

                // save the inserted mzid details with the project
                projectDetails.setContactFirstName(contactFirstNameJTextField.getText().trim());
                projectDetails.setContactLastName(contactLastNameJTextField.getText().trim());
                projectDetails.setContactEmail(contactEmailJTextField.getText().trim());
                projectDetails.setContactAddress(contactAddressJTextField.getText().trim());
                if (!contactUrlJTextField.getText().trim().isEmpty()) {
                    projectDetails.setContactUrl(contactUrlJTextField.getText().trim());
                } else {
                    projectDetails.setContactUrl(null);
                }

                projectDetails.setOrganizationName(organizationNameJTextField.getText().trim());
                projectDetails.setOrganizationEmail(organizationEmailJTextField.getText().trim());
                projectDetails.setOrganizationAddress(organizationAddressJTextField.getText().trim());
                if (!organizationUrlJTextField.getText().trim().isEmpty()) {
                    projectDetails.setOrganizationUrl(organizationUrlJTextField.getText().trim());
                } else {
                    projectDetails.setOrganizationUrl(null);
                }

                projectDetails.setIncludeProteinSequences(includeSequencesCheckBox.isSelected());
                projectDetails.setMzIdentOutputFile(outputFolderJTextField.getText());
                peptideShakerGUI.setDataSaved(false); 

                boolean conversionCompleted = false;

                // make sure that all annotations are included
                double currentIntensityLimit = annotationParameters.getAnnotationIntensityLimit();
                annotationParameters.setIntensityLimit(0.0);

                try {

                    FastaSummary fastaSummary = FastaSummary.getSummary(
                            projectDetails.getFastaFile(),
                            identificationParameters.getFastaParameters(),
                            progressDialog
                    );

                    MzIdentMLExport mzIdentMLExport = new MzIdentMLExport(
                            PeptideShaker.getVersion(),
                            peptideShakerGUI.getIdentification(),
                            projectDetails,
                            identificationParameters,
                            peptideShakerGUI.getSequenceProvider(),
                            peptideShakerGUI.getProteinDetailsProvider(),
                            peptideShakerGUI.getSpectrumProvider(),
                            ModificationFactory.getInstance(),
                            fastaSummary,
                            peptideShakerGUI.getIdentificationFeaturesGenerator(),
                            finalOutputFile,
                            includeSequencesCheckBox.isSelected(),
                            progressDialog,
                            true
                    );
                    mzIdentMLExport.createMzIdentMLFile(mzIdentMLVersion);

                    // validate the mzidentml file
                    if (validateMzIdentML && !progressDialog.isRunCanceled()) {
                        progressDialog.setPrimaryProgressCounterIndeterminate(true);
                        progressDialog.setTitle("Validating mzIdentML. Please Wait...");
                        String errors = validateMzIdentML(finalOutputFile);

                        // see if any errors were found, and display them to the user
                        if (!errors.isEmpty()) {
                            JOptionPane.showMessageDialog(
                                    null,
                                    errors,
                                    "mzIdentML Errors",
                                    JOptionPane.ERROR_MESSAGE
                            );
                        } else {
                            conversionCompleted = true;
                        }
                    } else {
                        conversionCompleted = true;
                    }

                } catch (Exception e) {
                    peptideShakerGUI.catchException(e);
                    progressDialog.setRunCanceled();
                    progressDialog.dispose();
                    return;
                } finally {
                    // reset the annotation level
                    annotationParameters.setIntensityLimit(currentIntensityLimit);
                }
                
                if (exportSdrf) {
                    
                    progressDialog.setPrimaryProgressCounterIndeterminate(true);
                    progressDialog.setTitle("Exporting sdrf draft. Please Wait...");

                    // Export an sdrf file next to the mzIdentML file
                    File sdrfFile = new File(IoUtil.removeExtension(finalOutputFile.getAbsolutePath()) + ".sdrf");

                    SdrfExport.writeSdrf(
                            sdrfFile,
                            peptideShakerGUI.getIdentificationParameters().getSearchParameters(),
                            peptideShakerGUI.getIdentification().getFractions(),
                            ModificationFactory.getInstance()
                    );
                
                }

                // close the progress dialog
                boolean processCancelled = progressDialog.isRunCanceled();
                progressDialog.setRunFinished();

                // display a conversion complete message to the user
                if (conversionCompleted && !processCancelled) {
                    JOptionPane.showMessageDialog(
                            MzIdentMLExportDialog.this,
                            JOptionEditorPane.getJOptionEditorPane(
                                    "mzIdentML file \'" + new File(outputFolderJTextField.getText()).getAbsolutePath() + "\' created."
                                    + "<br><br>"
                                    + "Review your mzIdentML files with <a href=\"https://github.com/PRIDE-Toolsuite/pride-inspector\">PRIDE Inspector</a>.<br>"
                                    + "Publish your mzIdentML files via <a href=\"http://www.proteomexchange.org/submission\">ProteomeXchange</a>."),
                            "File Created",
                            JOptionPane.INFORMATION_MESSAGE
                    );
                    dispose();
                }

                if (processCancelled) {
                    JOptionPane.showMessageDialog(
                            peptideShakerGUI, 
                            "mzIdentML conversion cancelled by the user.", 
                            "mzIdentML Conversion Cancelled", 
                            JOptionPane.WARNING_MESSAGE
                    );
                }
            }
        }.start();
    }//GEN-LAST:event_convertJButtonActionPerformed

    /**
     * @see #validateInput()
     */
    private void contactUrlJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_contactUrlJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_contactUrlJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void organizationNameJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_organizationNameJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_organizationNameJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void organizationEmailJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_organizationEmailJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_organizationEmailJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void organizationUrlJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_organizationUrlJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_organizationUrlJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void contactAddressJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_contactAddressJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_contactAddressJTextFieldKeyReleased

    /**
     * @see #validateInput()
     */
    private void organizationAddressJTextFieldKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_organizationAddressJTextFieldKeyReleased
        validateInput();
    }//GEN-LAST:event_organizationAddressJTextFieldKeyReleased

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel backgroundJPanel;
    private javax.swing.JButton browseOutputFolderJButton;
    private javax.swing.JTextField contactAddressJTextField;
    private javax.swing.JLabel contactAddressLabel;
    private javax.swing.JTextField contactEmailJTextField;
    private javax.swing.JLabel contactEmailLabel;
    private javax.swing.JTextField contactFirstNameJTextField;
    private javax.swing.JLabel contactFirstNameLabel;
    private javax.swing.JTextField contactLastNameJTextField;
    private javax.swing.JLabel contactLastNameLabel;
    private javax.swing.JPanel contactPanel;
    private javax.swing.JTextField contactUrlJTextField;
    private javax.swing.JLabel contactUrlLabel;
    private javax.swing.JButton convertJButton;
    private javax.swing.JLabel helpLabel;
    private javax.swing.JCheckBox includeSequencesCheckBox;
    private javax.swing.JButton openDialogHelpJButton;
    private javax.swing.JTextField organizationAddressJTextField;
    private javax.swing.JLabel organizationAddressLabel;
    private javax.swing.JTextField organizationEmailJTextField;
    private javax.swing.JLabel organizationEmailLabel;
    private javax.swing.JTextField organizationNameJTextField;
    private javax.swing.JLabel organizationNameLabel;
    private javax.swing.JPanel organizationPanel;
    private javax.swing.JTextField organizationUrlJTextField;
    private javax.swing.JLabel organizationUrlLabel;
    private javax.swing.JTextField outputFolderJTextField;
    private javax.swing.JLabel outputFolderLabel;
    private javax.swing.JPanel outputPanel;
    // End of variables declaration//GEN-END:variables

    /**
     * Validates that the required input has been inserted and enables or
     * disables the Convert button.
     */
    private void validateInput() {

        boolean inputValid = true;

        if (contactFirstNameJTextField.getText().length() == 0
                || contactLastNameJTextField.getText().length() == 0
                || contactEmailJTextField.getText().length() == 0
                || contactAddressJTextField.getText().length() == 0
                || organizationNameJTextField.getText().length() == 0
                || organizationEmailJTextField.getText().length() == 0
                || organizationAddressJTextField.getText().length() == 0
                || outputFolderJTextField.getText().length() == 0) {
            inputValid = false;
        }

        convertJButton.setEnabled(inputValid);

        // highlight the fields that have not been filled
        if (contactFirstNameJTextField.getText().length() > 0) {
            contactFirstNameLabel.setForeground(Color.BLACK);
        } else {
            contactFirstNameLabel.setForeground(Color.RED);
        }

        if (contactLastNameJTextField.getText().length() > 0) {
            contactLastNameLabel.setForeground(Color.BLACK);
        } else {
            contactLastNameLabel.setForeground(Color.RED);
        }

        if (contactEmailJTextField.getText().length() > 0) {
            contactEmailLabel.setForeground(Color.BLACK);
        } else {
            contactEmailLabel.setForeground(Color.RED);
        }

        if (contactAddressJTextField.getText().length() > 0) {
            contactAddressLabel.setForeground(Color.BLACK);
        } else {
            contactAddressLabel.setForeground(Color.RED);
        }

        if (organizationNameJTextField.getText().length() > 0) {
            organizationNameLabel.setForeground(Color.BLACK);
        } else {
            organizationNameLabel.setForeground(Color.RED);
        }

        if (organizationEmailJTextField.getText().length() > 0) {
            organizationEmailLabel.setForeground(Color.BLACK);
        } else {
            organizationEmailLabel.setForeground(Color.RED);
        }

        if (organizationAddressJTextField.getText().length() > 0) {
            organizationAddressLabel.setForeground(Color.BLACK);
        } else {
            organizationAddressLabel.setForeground(Color.RED);
        }

        if (outputFolderJTextField.getText().length() > 0) {
            outputFolderLabel.setForeground(Color.BLACK);
        } else {
            outputFolderLabel.setForeground(Color.RED);
        }
    }

    /**
     * Returns true if the user has canceled the progress.
     *
     * @return true if the user has canceled the progress
     */
    public boolean progressCancelled() {
        return progressDialog.isRunCanceled();
    }

    /**
     * Validates the mzIdentML file according to the schema.
     *
     * @param mzidFile the file to validate
     * @return the error messages, if any
     * @throws SAXException
     * @throws MalformedURLException
     * @throws FileNotFoundException
     * @throws URISyntaxException
     */
    private String validateMzIdentML(File mzidFile) throws SAXException, MalformedURLException, FileNotFoundException, URISyntaxException {

        String errors = "";

        GenericSchemaValidator genericValidator = new GenericSchemaValidator();
        genericValidator.setSchema(new URI("http://www.psidev.info/files/mzIdentML1.1.0.xsd"));

        ErrorHandlerIface handler = new ValidationErrorHandler();
        genericValidator.setErrorHandler(handler);
        BufferedReader br = new BufferedReader(new FileReader(mzidFile));
        genericValidator.validate(br);

        //noinspection unchecked
        List<String> errorMsgs = handler.getErrorMessages();
        if (!errorMsgs.isEmpty()) {
            int errorCount = 0;
            for (String error : errorMsgs) {
                errors += ++errorCount + ": " + error + "\n\n";
                System.out.println(errorCount + ": " + error);
            }
        }

        return errors;
    }
}
