package eu.isas.reporter.gui;

import com.compomics.util.experiment.biology.ions.impl.ElementaryIon;
import com.compomics.util.experiment.quantification.reporterion.ReporterMethodFactory;
import com.compomics.util.gui.file_handling.FileChooserUtil;
import java.io.File;
import java.io.IOException;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import org.xmlpull.v1.XmlPullParserException;

/**
 * Dialog for setting the reporter method details.
 *
 * @author Harald Barsnes
 */
public class MethodSettingsDialog extends javax.swing.JDialog {

    /**
     * The parent dialog.
     */
    private NewDialog newDialog;
    /**
     * The compomics reporter methods factory.
     */
    private ReporterMethodFactory methodsFactory = ReporterMethodFactory.getInstance();
    /**
     * Boolean indicating of the user has made changes to the table.
     */
    private boolean valuesChanged = false;

    /**
     * Creates a new MethodSettingsDialog.
     *
     * @param newDialog the parent dialog
     * @param modal if the dialog is to be modal or not
     */
    public MethodSettingsDialog(
            NewDialog newDialog,
            boolean modal
    ) {
        super(newDialog, modal);
        this.newDialog = newDialog;
        initComponents();
        setUpGui();
        setLocationRelativeTo(newDialog);
        setVisible(true);
    }

    /**
     * Set up the GUI.
     */
    private void setUpGui() {

        reagentsTableJScrollPane.getViewport().setOpaque(false);
        reagentsTable.getTableHeader().setReorderingAllowed(false);

        if (newDialog.getMethodsFile() != null) {
            txtConfigurationFileLocation.setText(newDialog.getMethodsFile().getAbsolutePath());
        } else {
            txtConfigurationFileLocation.setText("(not saved to file)");
        }

        reagentsTable.getColumnModel().getColumn(0).setMaxWidth(30);
        reagentsTable.getColumnModel().getColumn(1).setMaxWidth(100);

    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        backgroundPanel = new javax.swing.JPanel();
        isotopeCorrectionPanel = new javax.swing.JPanel();
        reagentsTableJScrollPane = new javax.swing.JScrollPane();
        reagentsTable = new javax.swing.JTable();
        configFilePanel = new javax.swing.JPanel();
        txtConfigurationFileLocation = new javax.swing.JTextField();
        browseConfigButton = new javax.swing.JButton();
        saveConfigButton = new javax.swing.JButton();
        okButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Reporter Method Settings");

        backgroundPanel.setBackground(new java.awt.Color(230, 230, 230));

        isotopeCorrectionPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Reagents"));
        isotopeCorrectionPanel.setOpaque(false);

        reagentsTable.setModel(new CorrectionTableModel());
        reagentsTable.setOpaque(false);
        reagentsTableJScrollPane.setViewportView(reagentsTable);

        javax.swing.GroupLayout isotopeCorrectionPanelLayout = new javax.swing.GroupLayout(isotopeCorrectionPanel);
        isotopeCorrectionPanel.setLayout(isotopeCorrectionPanelLayout);
        isotopeCorrectionPanelLayout.setHorizontalGroup(
            isotopeCorrectionPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(isotopeCorrectionPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(reagentsTableJScrollPane, javax.swing.GroupLayout.DEFAULT_SIZE, 804, Short.MAX_VALUE)
                .addContainerGap())
        );
        isotopeCorrectionPanelLayout.setVerticalGroup(
            isotopeCorrectionPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(isotopeCorrectionPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(reagentsTableJScrollPane, javax.swing.GroupLayout.DEFAULT_SIZE, 308, Short.MAX_VALUE)
                .addContainerGap())
        );

        configFilePanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Settings File"));
        configFilePanel.setOpaque(false);

        txtConfigurationFileLocation.setEditable(false);

        browseConfigButton.setText("Browse");
        browseConfigButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                browseConfigButtonActionPerformed(evt);
            }
        });

        saveConfigButton.setText("Save As");
        saveConfigButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveConfigButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout configFilePanelLayout = new javax.swing.GroupLayout(configFilePanel);
        configFilePanel.setLayout(configFilePanelLayout);
        configFilePanelLayout.setHorizontalGroup(
            configFilePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(configFilePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(txtConfigurationFileLocation)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(browseConfigButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(saveConfigButton)
                .addContainerGap())
        );
        configFilePanelLayout.setVerticalGroup(
            configFilePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(configFilePanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(configFilePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(txtConfigurationFileLocation, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(saveConfigButton)
                    .addComponent(browseConfigButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout backgroundPanelLayout = new javax.swing.GroupLayout(backgroundPanel);
        backgroundPanel.setLayout(backgroundPanelLayout);
        backgroundPanelLayout.setHorizontalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(configFilePanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(isotopeCorrectionPanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, backgroundPanelLayout.createSequentialGroup()
                        .addGap(0, 0, Short.MAX_VALUE)
                        .addComponent(okButton, javax.swing.GroupLayout.PREFERRED_SIZE, 66, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addContainerGap())
        );
        backgroundPanelLayout.setVerticalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(configFilePanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(isotopeCorrectionPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(okButton)
                .addContainerGap())
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Opens a file chooser for importing method settings.
     *
     * @param evt
     */
    private void browseConfigButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_browseConfigButtonActionPerformed

        if (txtConfigurationFileLocation.getText().length() > 0) {
            newDialog.getReporterGui().getLastSelectedFolder().setLastSelectedFolder(txtConfigurationFileLocation.getText());
        }

        File selectedFile = FileChooserUtil.getUserSelectedFile(
                this,
                ".xml",
                "Reporter Method File (*.xml)",
                "Select Settings File",
                newDialog.getReporterGui().getLastSelectedFolder().getLastSelectedFolder(),
                null,
                true
        );

        if (selectedFile != null) {

            try {

                methodsFactory.importMethods(selectedFile);
                newDialog.setSelectedMethod(methodsFactory.getReporterMethod(newDialog.getSelectedMethod().getName()));
                newDialog.setReagents(newDialog.getSelectedMethod().getReagentsSortedByMass());
                refresh();
                txtConfigurationFileLocation.setText(selectedFile.getAbsolutePath());
                newDialog.setMethodsFile(selectedFile);
                newDialog.getReporterGui().getLastSelectedFolder().setLastSelectedFolder(selectedFile.getPath());

            } catch (IOException e) {

                JOptionPane.showMessageDialog(
                        null,
                        "File " + selectedFile.getAbsolutePath() + " not found.",
                        "File Not Found",
                        JOptionPane.WARNING_MESSAGE
                );

            } catch (XmlPullParserException e) {

                JOptionPane.showMessageDialog(
                        null,
                        "An error occurred while parsing "
                        + selectedFile.getAbsolutePath()
                        + " at line " + e.getLineNumber() + ".",
                        "Parsing Error",
                        JOptionPane.WARNING_MESSAGE
                );

                e.printStackTrace();
            }
        }
    }//GEN-LAST:event_browseConfigButtonActionPerformed

    /**
     * Save the method settings.
     *
     * @param evt
     */
    private void saveConfigButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveConfigButtonActionPerformed

        String tempFileName = null;

        if (txtConfigurationFileLocation.getText().length() > 0) {
            newDialog.getReporterGui().getLastSelectedFolder().setLastSelectedFolder(txtConfigurationFileLocation.getText());
            tempFileName = new File(txtConfigurationFileLocation.getText()).getName();
        }

        File selectedFile = FileChooserUtil.getUserSelectedFile(
                this,
                ".xml",
                "Reporter Method File (*.xml)",
                "Save Settings File",
                newDialog.getReporterGui().getLastSelectedFolder().getLastSelectedFolder(),
                tempFileName,
                false
        );

        if (selectedFile != null) {

            try {

                methodsFactory.saveFile(selectedFile);
                newDialog.setMethodsFile(selectedFile);
                txtConfigurationFileLocation.setText(selectedFile.getAbsolutePath());
                valuesChanged = false;

                JOptionPane.showMessageDialog(
                        null,
                        "Settings saved to " + selectedFile.getAbsolutePath() + ".",
                        "Settings Saved",
                        JOptionPane.INFORMATION_MESSAGE
                );

            } catch (IOException ex) {

                JOptionPane.showMessageDialog(
                        null,
                        "An error occured when saving the file.",
                        "File Error",
                        JOptionPane.WARNING_MESSAGE
                );

                ex.printStackTrace();
            }
        }
    }//GEN-LAST:event_saveConfigButtonActionPerformed

    /**
     * Close the dialog.
     *
     * @param evt
     */
    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed

        if (valuesChanged) {

            int selection = JOptionPane.showConfirmDialog(
                    this,
                    "Do you want to save the new settings to a file?",
                    "Save File?",
                    JOptionPane.YES_NO_CANCEL_OPTION
            );

            switch (selection) {
                case JOptionPane.YES_OPTION:
                    saveConfigButtonActionPerformed(null);
                    break;
                case JOptionPane.CANCEL_OPTION:
                    return;
                case JOptionPane.NO_OPTION:
                    newDialog.setMethodsFile(null);
                    break;
                default:
                    break;
            }
        }

        newDialog.updateReagentNames();
        dispose();
    }//GEN-LAST:event_okButtonActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel backgroundPanel;
    private javax.swing.JButton browseConfigButton;
    private javax.swing.JPanel configFilePanel;
    private javax.swing.JPanel isotopeCorrectionPanel;
    private javax.swing.JButton okButton;
    private javax.swing.JTable reagentsTable;
    private javax.swing.JScrollPane reagentsTableJScrollPane;
    private javax.swing.JButton saveConfigButton;
    private javax.swing.JTextField txtConfigurationFileLocation;
    // End of variables declaration//GEN-END:variables

    /**
     * Updates the combo box and table values based on the currently selected
     * quantification method.
     */
    private void refresh() {

        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                reagentsTable.revalidate();
                reagentsTable.repaint();
            }
        });
    }

    /**
     * Table model for the correction factors table.
     */
    private class CorrectionTableModel extends DefaultTableModel {

        @Override
        public int getRowCount() {
            return newDialog.getReagents().size();
        }

        @Override
        public int getColumnCount() {
            return 8;
        }

        @Override
        public String getColumnName(int column) {

            switch (column) {
                case 0:
                    return " ";
                case 1:
                    return "Name";
                case 2:
                    return "Mass";
                case 3:
                    return "-2 C13 [%]";
                case 4:
                    return "-1 C13 [%]";
                case 5:
                    return "Monoisotopic [%]";
                case 6:
                    return "+1 C13 [%]";
                case 7:
                    return "+2 C13 [%]";
                default:
                    return "";
            }

        }

        @Override
        public Object getValueAt(int row, int column) {

            String reagentName = newDialog.getReagents().get(row);

            switch (column) {
                case 0:
                    return row + 1;
                case 1:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getReporterIon().getName();
                case 2:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getReporterIon().getTheoreticMass() + ElementaryIon.proton.getTheoreticMass();
                case 3:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getMinus2();
                case 4:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getMinus1();
                case 5:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getRef();
                case 6:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getPlus1();
                case 7:
                    return newDialog.getSelectedMethod().getReagent(reagentName).getPlus2();
                default:
                    return "";
            }

        }

        @Override
        public void setValueAt(Object aValue, int row, int column) {

            String reagentName = newDialog.getReagents().get(row);

            switch (column) {
                case 1:
                    newDialog.getSelectedMethod().getReagent(reagentName).getReporterIon().setName((String) aValue);
                    valuesChanged = true;
                    break;
                case 2:
                    newDialog.getSelectedMethod().getReagent(reagentName).getReporterIon().setTheoreticMass((Double) aValue);
                    valuesChanged = true;
                    break;
                case 3:
                    newDialog.getSelectedMethod().getReagent(reagentName).setMinus2((Double) aValue);
                    valuesChanged = true;
                    break;
                case 4:
                    newDialog.getSelectedMethod().getReagent(reagentName).setMinus1((Double) aValue);
                    valuesChanged = true;
                    break;
                case 5:
                    newDialog.getSelectedMethod().getReagent(reagentName).setRef((Double) aValue);
                    valuesChanged = true;
                    break;
                case 6:
                    newDialog.getSelectedMethod().getReagent(reagentName).setPlus1((Double) aValue);
                    valuesChanged = true;
                    break;
                case 7:
                    newDialog.getSelectedMethod().getReagent(reagentName).setPlus2((Double) aValue);
                    valuesChanged = true;
                    break;
                default:
                    break;
            }

        }

        @Override
        public Class getColumnClass(int columnIndex) {
            return getValueAt(0, columnIndex).getClass();
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            return column != 0;
        }
    }
}
