package eu.isas.reporter.gui.settings.quantification;

import com.compomics.util.gui.renderers.AlignedListCellRenderer;
import eu.isas.reporter.calculation.normalization.NormalizationType;
import eu.isas.reporter.settings.NormalizationSettings;
import java.io.File;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.SwingConstants;
import javax.swing.filechooser.FileFilter;

/**
 * The preferences dialog.
 *
 * @author Marc Vaudel
 * @author Harald Barsnes
 */
public class NormalizationSettingsDialog extends javax.swing.JDialog {

    /**
     * Boolean indicating whether the editing of the settings has been canceled.
     */
    private boolean canceled = false;
    /**
     * The selected PSM normalization type.
     */
    private NormalizationType psmNormalizationType;
    /**
     * The selected peptide normalization type.
     */
    private NormalizationType peptideNormalizationType;
    /**
     * The selected protein normalization type.
     */
    private NormalizationType proteinNormalizationType;
    /**
     * FASTA file containing the stable proteins.
     */
    private File stableProteinsFile;
    /**
     * FASTA file containing the contaminant proteins.
     */
    private File contaminantsFile;

    /**
     * Creates a new NormalizationSettingsDialog.
     *
     * @param parentDialog the parent dialog
     * @param normalizationSettings the settings to display
     * @param editable boolean indicating whether the settings can be edited
     */
    public NormalizationSettingsDialog(
            JDialog parentDialog,
            NormalizationSettings normalizationSettings,
            boolean editable
    ) {

        super(parentDialog, true);
        initComponents();
        setUpGui(editable);
        populateGUI(normalizationSettings);
        setLocationRelativeTo(parentDialog);
        setVisible(true);

    }

    /**
     * Set up the GUI.
     *
     * @param editable boolean indicating whether the settings can be edited
     */
    private void setUpGui(boolean editable) {

        psmNormalizationCmb.setEnabled(editable);
        peptideNormalizationCmb.setEnabled(editable);
        proteinNormalizationCmb.setEnabled(editable);
        stableProteinsBrowseButton.setEnabled(editable);
        stableProteinsClearButton.setEnabled(editable);
        contaminantsBrowseButton.setEnabled(editable);
        contaminantsClearButton.setEnabled(editable);

        psmNormalizationCmb.setRenderer(new AlignedListCellRenderer(SwingConstants.CENTER));
        peptideNormalizationCmb.setRenderer(new AlignedListCellRenderer(SwingConstants.CENTER));
        proteinNormalizationCmb.setRenderer(new AlignedListCellRenderer(SwingConstants.CENTER));

    }

    /**
     * Fills the GUI with the given settings.
     *
     * @param ratioEstimationSettings the normalization settings to display
     */
    private void populateGUI(NormalizationSettings normalizationSettings) {

        psmNormalizationType = normalizationSettings.getPsmNormalization();
        peptideNormalizationType = normalizationSettings.getPeptideNormalization();
        proteinNormalizationType = normalizationSettings.getProteinNormalization();
        stableProteinsFile = normalizationSettings.getStableProteinsFastaFile();
        contaminantsFile = normalizationSettings.getContaminantsFastaFile();

        updateGUI();

    }

    /**
     * Updates the GUI according to the dialog attributes.
     */
    private void updateGUI() {

        psmNormalizationCmb.setSelectedItem(psmNormalizationType);
        peptideNormalizationCmb.setSelectedItem(peptideNormalizationType);
        proteinNormalizationCmb.setSelectedItem(proteinNormalizationType);

        if (stableProteinsFile != null) {
            stableProteinsTxt.setText(stableProteinsFile.getName());
        } else {
            stableProteinsTxt.setText("No File Selected");
        }

        if (contaminantsFile != null) {
            contaminantsTxt.setText(contaminantsFile.getName());
        } else {
            contaminantsTxt.setText("No File Selected");
        }

    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        backgroundPanel = new javax.swing.JPanel();
        cancelButton = new javax.swing.JButton();
        okButton = new javax.swing.JButton();
        matchesNormalizationPanel = new javax.swing.JPanel();
        psmNormalizationLbl = new javax.swing.JLabel();
        psmNormalizationCmb = new javax.swing.JComboBox();
        peptideNormalizationLbl = new javax.swing.JLabel();
        peptideNormalizationCmb = new javax.swing.JComboBox();
        proteinNormalizationLbl = new javax.swing.JLabel();
        proteinNormalizationCmb = new javax.swing.JComboBox();
        specialProteinsPanel = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        stableProteinsClearButton = new javax.swing.JButton();
        stableProteinsBrowseButton = new javax.swing.JButton();
        contaminantsClearButton = new javax.swing.JButton();
        contaminantsBrowseButton = new javax.swing.JButton();
        stableProteinsTxt = new javax.swing.JTextField();
        contaminantsTxt = new javax.swing.JTextField();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Normalization Settings");
        setResizable(false);

        backgroundPanel.setBackground(new java.awt.Color(230, 230, 230));

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        matchesNormalizationPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Matches Normalization"));
        matchesNormalizationPanel.setOpaque(false);

        psmNormalizationLbl.setText("PSM Normalization");

        psmNormalizationCmb.setModel(new DefaultComboBoxModel(NormalizationType.values()));
        psmNormalizationCmb.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                psmNormalizationCmbActionPerformed(evt);
            }
        });

        peptideNormalizationLbl.setText("Peptide Normalization");

        peptideNormalizationCmb.setModel(new DefaultComboBoxModel(NormalizationType.values()));
        peptideNormalizationCmb.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                peptideNormalizationCmbActionPerformed(evt);
            }
        });

        proteinNormalizationLbl.setText("Protein Normalization");

        proteinNormalizationCmb.setModel(new DefaultComboBoxModel(NormalizationType.values()));
        proteinNormalizationCmb.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                proteinNormalizationCmbActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout matchesNormalizationPanelLayout = new javax.swing.GroupLayout(matchesNormalizationPanel);
        matchesNormalizationPanel.setLayout(matchesNormalizationPanelLayout);
        matchesNormalizationPanelLayout.setHorizontalGroup(
            matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(matchesNormalizationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addGroup(matchesNormalizationPanelLayout.createSequentialGroup()
                        .addComponent(psmNormalizationLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(psmNormalizationCmb, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                    .addGroup(matchesNormalizationPanelLayout.createSequentialGroup()
                        .addComponent(proteinNormalizationLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(proteinNormalizationCmb, javax.swing.GroupLayout.PREFERRED_SIZE, 380, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(matchesNormalizationPanelLayout.createSequentialGroup()
                        .addComponent(peptideNormalizationLbl, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(peptideNormalizationCmb, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)))
                .addContainerGap(159, Short.MAX_VALUE))
        );
        matchesNormalizationPanelLayout.setVerticalGroup(
            matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(matchesNormalizationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(psmNormalizationLbl)
                    .addComponent(psmNormalizationCmb, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(peptideNormalizationLbl)
                    .addComponent(peptideNormalizationCmb, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(matchesNormalizationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(proteinNormalizationLbl)
                    .addComponent(proteinNormalizationCmb, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(14, Short.MAX_VALUE))
        );

        specialProteinsPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Special Proteins"));
        specialProteinsPanel.setOpaque(false);

        jLabel1.setText("Stable Proteins");

        jLabel2.setText("Contaminants");

        stableProteinsClearButton.setText("Clear");
        stableProteinsClearButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                stableProteinsClearButtonActionPerformed(evt);
            }
        });

        stableProteinsBrowseButton.setText("Browse");
        stableProteinsBrowseButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                stableProteinsBrowseButtonActionPerformed(evt);
            }
        });

        contaminantsClearButton.setText("Clear");
        contaminantsClearButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                contaminantsClearButtonActionPerformed(evt);
            }
        });

        contaminantsBrowseButton.setText("Browse");
        contaminantsBrowseButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                contaminantsBrowseButtonActionPerformed(evt);
            }
        });

        stableProteinsTxt.setEditable(false);
        stableProteinsTxt.setHorizontalAlignment(javax.swing.JTextField.CENTER);

        contaminantsTxt.setEditable(false);
        contaminantsTxt.setHorizontalAlignment(javax.swing.JTextField.CENTER);

        javax.swing.GroupLayout specialProteinsPanelLayout = new javax.swing.GroupLayout(specialProteinsPanel);
        specialProteinsPanel.setLayout(specialProteinsPanelLayout);
        specialProteinsPanelLayout.setHorizontalGroup(
            specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, specialProteinsPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel1, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel2, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(18, 18, 18)
                .addGroup(specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(contaminantsTxt, javax.swing.GroupLayout.DEFAULT_SIZE, 383, Short.MAX_VALUE)
                    .addComponent(stableProteinsTxt, javax.swing.GroupLayout.DEFAULT_SIZE, 383, Short.MAX_VALUE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, specialProteinsPanelLayout.createSequentialGroup()
                        .addComponent(stableProteinsBrowseButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(stableProteinsClearButton, javax.swing.GroupLayout.PREFERRED_SIZE, 67, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, specialProteinsPanelLayout.createSequentialGroup()
                        .addComponent(contaminantsBrowseButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(contaminantsClearButton, javax.swing.GroupLayout.PREFERRED_SIZE, 67, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addContainerGap())
        );
        specialProteinsPanelLayout.setVerticalGroup(
            specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(specialProteinsPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(stableProteinsClearButton)
                    .addComponent(stableProteinsBrowseButton)
                    .addComponent(stableProteinsTxt, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(specialProteinsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel2)
                    .addComponent(contaminantsClearButton)
                    .addComponent(contaminantsBrowseButton)
                    .addComponent(contaminantsTxt, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout backgroundPanelLayout = new javax.swing.GroupLayout(backgroundPanel);
        backgroundPanel.setLayout(backgroundPanelLayout);
        backgroundPanelLayout.setHorizontalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(backgroundPanelLayout.createSequentialGroup()
                        .addGap(0, 585, Short.MAX_VALUE)
                        .addComponent(okButton, javax.swing.GroupLayout.PREFERRED_SIZE, 69, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(cancelButton, javax.swing.GroupLayout.PREFERRED_SIZE, 69, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(matchesNormalizationPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(specialProteinsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );
        backgroundPanelLayout.setVerticalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(matchesNormalizationPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(specialProteinsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(cancelButton)
                    .addComponent(okButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Close the dialog and cancel the changes.
     *
     * @param evt
     */
    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        canceled = true;
        dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed

    /**
     * Save the data and close the dialog.
     *
     * @param evt
     */
    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed
        if (validateInput()) {
            dispose();
        }
    }//GEN-LAST:event_okButtonActionPerformed

    /**
     * Clear the list of stable proteins.
     *
     * @param evt
     */
    private void stableProteinsClearButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stableProteinsClearButtonActionPerformed

        stableProteinsFile = null;
        updateGUI();

    }//GEN-LAST:event_stableProteinsClearButtonActionPerformed

    /**
     * Clear the list of contaminants.
     *
     * @param evt
     */
    private void contaminantsClearButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_contaminantsClearButtonActionPerformed

        contaminantsFile = null;
        updateGUI();

    }//GEN-LAST:event_contaminantsClearButtonActionPerformed

    /**
     * Browse for a FASTA file of stable proteins.
     *
     * @param evt
     */
    private void stableProteinsBrowseButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stableProteinsBrowseButtonActionPerformed

        File startFile = null;

        if (stableProteinsFile != null) {
            startFile = stableProteinsFile.getParentFile();
        }

        File fastaFile = getFastaFile(startFile);
        stableProteinsFile = fastaFile;

        updateGUI();

    }//GEN-LAST:event_stableProteinsBrowseButtonActionPerformed

    /**
     * Browse for a FASTA file of contaminants.
     *
     * @param evt
     */
    private void contaminantsBrowseButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_contaminantsBrowseButtonActionPerformed

        File startFile = null;

        if (contaminantsFile != null) {
            startFile = contaminantsFile.getParentFile();
        }

        File fastaFile = getFastaFile(startFile);

        contaminantsFile = fastaFile;
        updateGUI();

    }//GEN-LAST:event_contaminantsBrowseButtonActionPerformed

    /**
     * Update the PSM normalization type.
     *
     * @param evt
     */
    private void psmNormalizationCmbActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_psmNormalizationCmbActionPerformed
        psmNormalizationType = (NormalizationType) psmNormalizationCmb.getSelectedItem();
    }//GEN-LAST:event_psmNormalizationCmbActionPerformed

    /**
     * Update the peptide normalization type.
     *
     * @param evt
     */
    private void peptideNormalizationCmbActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_peptideNormalizationCmbActionPerformed
        peptideNormalizationType = (NormalizationType) peptideNormalizationCmb.getSelectedItem();
    }//GEN-LAST:event_peptideNormalizationCmbActionPerformed

    /**
     * Update the protein normalization type.
     *
     * @param evt
     */
    private void proteinNormalizationCmbActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_proteinNormalizationCmbActionPerformed
        proteinNormalizationType = (NormalizationType) proteinNormalizationCmb.getSelectedItem();
    }//GEN-LAST:event_proteinNormalizationCmbActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel backgroundPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JButton contaminantsBrowseButton;
    private javax.swing.JButton contaminantsClearButton;
    private javax.swing.JTextField contaminantsTxt;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JPanel matchesNormalizationPanel;
    private javax.swing.JButton okButton;
    private javax.swing.JComboBox peptideNormalizationCmb;
    private javax.swing.JLabel peptideNormalizationLbl;
    private javax.swing.JComboBox proteinNormalizationCmb;
    private javax.swing.JLabel proteinNormalizationLbl;
    private javax.swing.JComboBox psmNormalizationCmb;
    private javax.swing.JLabel psmNormalizationLbl;
    private javax.swing.JPanel specialProteinsPanel;
    private javax.swing.JButton stableProteinsBrowseButton;
    private javax.swing.JButton stableProteinsClearButton;
    private javax.swing.JTextField stableProteinsTxt;
    // End of variables declaration//GEN-END:variables

    /**
     * Indicates whether the user canceled the editing.
     *
     * @return a boolean indicating whether the user canceled the editing
     */
    public boolean isCanceled() {
        return canceled;
    }

    /**
     * Methods which validates the user input (returns false in case of wrong
     * input).
     *
     * @return true if the input can be processed
     */
    private boolean validateInput() {
        return true;
    }

    /**
     * Returns the settings as set by the user.
     *
     * @return the settings as set by the user
     */
    public NormalizationSettings getNormalizationSettings() {

        NormalizationSettings normalizationSettings = new NormalizationSettings();
        normalizationSettings.setPsmNormalization(psmNormalizationType);
        normalizationSettings.setPeptideNormalization(peptideNormalizationType);
        normalizationSettings.setProteinNormalization(proteinNormalizationType);
        normalizationSettings.setStableProteinsFastaFile(stableProteinsFile);
        normalizationSettings.setContaminantsFastaFile(contaminantsFile);

        return normalizationSettings;

    }

    /**
     * Returns a FASTA file selected by the user.
     *
     * @param startLocation the folder to start browsing from
     *
     * @return a FASTA file selected by the user
     */
    private File getFastaFile(File startLocation) {

        JFileChooser fc = new JFileChooser(startLocation);

        FileFilter filter = new FileFilter() {
            @Override
            public boolean accept(File myFile) {
                return myFile.getName().toLowerCase().endsWith(".fasta")
                        || myFile.getName().toLowerCase().endsWith(".fas")
                        || myFile.isDirectory();
            }

            @Override
            public String getDescription() {
                return "FASTA (.fasta or .fas)";
            }
        };

        fc.setFileFilter(filter);
        int result = fc.showOpenDialog(this);

        if (result == JFileChooser.APPROVE_OPTION) {
            File file = fc.getSelectedFile();
            return file;
        }

        return null;

    }
}
