package eu.isas.reporter.gui.settings.quantification;

import com.compomics.util.experiment.quantification.reporterion.ReporterMethod;
import com.compomics.util.gui.renderers.AlignedListCellRenderer;
import eu.isas.reporter.settings.ReporterIonSelectionSettings;
import eu.isas.reporter.settings.ReporterIonsLocationType;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.SwingConstants;

/**
 * The preferences dialog.
 *
 * @author Marc Vaudel
 * @author Harald Barsnes
 */
public class ReporterIonSelectionSettingsDialog extends javax.swing.JDialog {

    /**
     * Boolean indicating whether the editing of the settings has been canceled.
     */
    private boolean canceled = false;
    /**
     * The reporter method used.
     */
    private ReporterMethod reporterMethod = null;

    /**
     * Creates a new ReporterIonSelectionSettingsDialog.
     *
     * @param parentDialog the parent dialog
     * @param reporterIonSelectionSettings the settings to display
     * @param reporterMethod the reporter method used
     * @param editable boolean indicating whether the settings can be edited
     */
    public ReporterIonSelectionSettingsDialog(
            JDialog parentDialog,
            ReporterIonSelectionSettings reporterIonSelectionSettings,
            ReporterMethod reporterMethod,
            boolean editable
    ) {

        super(parentDialog, true);
        this.reporterMethod = reporterMethod;
        initComponents();
        setUpGui(editable);
        populateGUI(reporterIonSelectionSettings);
        setLocationRelativeTo(parentDialog);
        setVisible(true);

    }

    /**
     * Creates a new ReporterIonSelectionSettingsDialog. The dialog will not
     * check whether the reporter ion m/z tolerance is sufficient to distinguish
     * ions.
     *
     * @param parentDialog the parent dialog
     * @param reporterIonSelectionSettings the settings to display
     * @param editable boolean indicating whether the settings can be edited
     */
    public ReporterIonSelectionSettingsDialog(
            JDialog parentDialog,
            ReporterIonSelectionSettings reporterIonSelectionSettings,
            boolean editable
    ) {

        this(
                parentDialog,
                reporterIonSelectionSettings,
                null,
                editable
        );

    }

    /**
     * Set up the GUI.
     *
     * @param editable boolean indicating whether the settings can be edited
     */
    private void setUpGui(boolean editable) {

        //@TODO: Set editable or not
        ionSelectionComboBox.setRenderer(new AlignedListCellRenderer(SwingConstants.CENTER));
    }

    /**
     * Fills the GUI with the given settings.
     *
     * @param reporterIonSelectionSettings the settings to display
     */
    private void populateGUI(ReporterIonSelectionSettings reporterIonSelectionSettings) {

        ionToleranceTxt.setText(reporterIonSelectionSettings.getReporterIonsMzTolerance() + "");

        if (reporterIonSelectionSettings.isMostAccurate()) {
            ionSelectionComboBox.setSelectedIndex(0);
        } else {
            ionSelectionComboBox.setSelectedIndex(1);
        }

        switch (reporterIonSelectionSettings.getReporterIonsLocation()) {

            case ms2Spectra:

                ms2SpectrumReportersOption.setSelected(true);
                break;

            case ms3Spectra:

                ms3SpectrumReportersOption.setSelected(true);
                break;

            case precursorMatching:

                precursorMatchingReportersOption.setSelected(true);
                mzTolTxt.setText(reporterIonSelectionSettings.getPrecursorMzTolerance() + "");
                rtTolTxt.setText(reporterIonSelectionSettings.getPrecursorRTTolerance() + "");
                break;

            default:
                break;
        }

        updateReporterLocationOptions();

    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        reporterLocationButtonGroup = new javax.swing.ButtonGroup();
        backgroundPanel = new javax.swing.JPanel();
        okButton = new javax.swing.JButton();
        spectrumAnalysisPanel = new javax.swing.JPanel();
        reporterIonMzToleranceLabel = new javax.swing.JLabel();
        ionToleranceTxt = new javax.swing.JTextField();
        ionSelectionLabel = new javax.swing.JLabel();
        ionSelectionComboBox = new javax.swing.JComboBox();
        reporterLocationPanel = new javax.swing.JPanel();
        ms2SpectrumReportersOption = new javax.swing.JRadioButton();
        precursorMatchingReportersOption = new javax.swing.JRadioButton();
        mzToleranceLabel = new javax.swing.JLabel();
        mzTolTxt = new javax.swing.JTextField();
        ppmCmb = new javax.swing.JComboBox();
        rtToleranceLabel = new javax.swing.JLabel();
        rtTolTxt = new javax.swing.JTextField();
        ms3SpectrumReportersOption = new javax.swing.JRadioButton();
        cancelButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Reporter Ion Selection Settings");
        setResizable(false);

        backgroundPanel.setBackground(new java.awt.Color(230, 230, 230));

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        spectrumAnalysisPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Spectrum Analysis"));
        spectrumAnalysisPanel.setOpaque(false);

        reporterIonMzToleranceLabel.setText("Reporter Tolerance (m/z)");

        ionToleranceTxt.setHorizontalAlignment(javax.swing.JTextField.CENTER);

        ionSelectionLabel.setText("Ion Selection");

        ionSelectionComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Most Accurate", "Most Intense" }));

        javax.swing.GroupLayout spectrumAnalysisPanelLayout = new javax.swing.GroupLayout(spectrumAnalysisPanel);
        spectrumAnalysisPanel.setLayout(spectrumAnalysisPanelLayout);
        spectrumAnalysisPanelLayout.setHorizontalGroup(
            spectrumAnalysisPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(spectrumAnalysisPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(spectrumAnalysisPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(spectrumAnalysisPanelLayout.createSequentialGroup()
                        .addComponent(reporterIonMzToleranceLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(ionToleranceTxt))
                    .addGroup(spectrumAnalysisPanelLayout.createSequentialGroup()
                        .addComponent(ionSelectionLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(ionSelectionComboBox, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)))
                .addContainerGap())
        );
        spectrumAnalysisPanelLayout.setVerticalGroup(
            spectrumAnalysisPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(spectrumAnalysisPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(spectrumAnalysisPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(reporterIonMzToleranceLabel)
                    .addComponent(ionToleranceTxt, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(spectrumAnalysisPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(ionSelectionLabel)
                    .addComponent(ionSelectionComboBox, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        reporterLocationPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Reporter Location"));
        reporterLocationPanel.setOpaque(false);

        reporterLocationButtonGroup.add(ms2SpectrumReportersOption);
        ms2SpectrumReportersOption.setText("MS2 Spectrum");
        ms2SpectrumReportersOption.setIconTextGap(10);
        ms2SpectrumReportersOption.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ms2SpectrumReportersOptionActionPerformed(evt);
            }
        });

        reporterLocationButtonGroup.add(precursorMatchingReportersOption);
        precursorMatchingReportersOption.setText("Precursor Matching");
        precursorMatchingReportersOption.setIconTextGap(10);
        precursorMatchingReportersOption.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                precursorMatchingReportersOptionReportersOptionActionPerformed(evt);
            }
        });

        mzToleranceLabel.setText("m/z tolerance");

        mzTolTxt.setHorizontalAlignment(javax.swing.JTextField.CENTER);

        ppmCmb.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "ppm", "m/z" }));

        rtToleranceLabel.setText("RT tolerance (s)");

        rtTolTxt.setHorizontalAlignment(javax.swing.JTextField.CENTER);
        rtTolTxt.setText("10");
        rtTolTxt.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rtTolTxtActionPerformed(evt);
            }
        });

        reporterLocationButtonGroup.add(ms3SpectrumReportersOption);
        ms3SpectrumReportersOption.setSelected(true);
        ms3SpectrumReportersOption.setText("MS3 Spectrum");
        ms3SpectrumReportersOption.setIconTextGap(10);
        ms3SpectrumReportersOption.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ms3SpectrumReportersOptionActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout reporterLocationPanelLayout = new javax.swing.GroupLayout(reporterLocationPanel);
        reporterLocationPanel.setLayout(reporterLocationPanelLayout);
        reporterLocationPanelLayout.setHorizontalGroup(
            reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(reporterLocationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(ms2SpectrumReportersOption, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(precursorMatchingReportersOption, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGroup(reporterLocationPanelLayout.createSequentialGroup()
                        .addGap(27, 27, 27)
                        .addGroup(reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                            .addComponent(rtToleranceLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 144, Short.MAX_VALUE)
                            .addComponent(mzToleranceLabel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(rtTolTxt, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addGroup(reporterLocationPanelLayout.createSequentialGroup()
                                .addComponent(mzTolTxt, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(ppmCmb, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))))
                    .addComponent(ms3SpectrumReportersOption, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        reporterLocationPanelLayout.setVerticalGroup(
            reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(reporterLocationPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(ms2SpectrumReportersOption)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(ms3SpectrumReportersOption)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(precursorMatchingReportersOption)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(mzToleranceLabel)
                    .addComponent(mzTolTxt, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(ppmCmb, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, 0)
                .addGroup(reporterLocationPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(rtToleranceLabel)
                    .addComponent(rtTolTxt, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout backgroundPanelLayout = new javax.swing.GroupLayout(backgroundPanel);
        backgroundPanel.setLayout(backgroundPanelLayout);
        backgroundPanelLayout.setHorizontalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(reporterLocationPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(spectrumAnalysisPanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, backgroundPanelLayout.createSequentialGroup()
                        .addGap(0, 0, Short.MAX_VALUE)
                        .addComponent(okButton, javax.swing.GroupLayout.PREFERRED_SIZE, 69, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(cancelButton, javax.swing.GroupLayout.PREFERRED_SIZE, 69, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addContainerGap())
        );
        backgroundPanelLayout.setVerticalGroup(
            backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(backgroundPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(spectrumAnalysisPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(reporterLocationPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(backgroundPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(okButton)
                    .addComponent(cancelButton))
                .addGap(9, 9, 9))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(backgroundPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Save the data and close the dialog.
     *
     * @param evt
     */
    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed
        if (validateInput()) {
            dispose();
        }
    }//GEN-LAST:event_okButtonActionPerformed

    /**
     * Update the reporter location options.
     *
     * @param evt
     */
    private void ms2SpectrumReportersOptionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ms2SpectrumReportersOptionActionPerformed
        updateReporterLocationOptions();
    }//GEN-LAST:event_ms2SpectrumReportersOptionActionPerformed

    /**
     * Update the reporter location options.
     *
     * @param evt
     */
    private void precursorMatchingReportersOptionReportersOptionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_precursorMatchingReportersOptionReportersOptionActionPerformed
        updateReporterLocationOptions();
    }//GEN-LAST:event_precursorMatchingReportersOptionReportersOptionActionPerformed

    /**
     * Validate the RT input.
     *
     * @param evt
     */
    private void rtTolTxtActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rtTolTxtActionPerformed
        // @TODO: validate the input
    }//GEN-LAST:event_rtTolTxtActionPerformed

    /**
     * Close the dialog and cancel the changes.
     *
     * @param evt
     */
    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed

        canceled = true;
        dispose();

    }//GEN-LAST:event_cancelButtonActionPerformed

    /**
     * Update the reporter location options.
     *
     * @param evt
     */
    private void ms3SpectrumReportersOptionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_ms3SpectrumReportersOptionActionPerformed
        updateReporterLocationOptions();
    }//GEN-LAST:event_ms3SpectrumReportersOptionActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel backgroundPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JComboBox ionSelectionComboBox;
    private javax.swing.JLabel ionSelectionLabel;
    private javax.swing.JTextField ionToleranceTxt;
    private javax.swing.JRadioButton ms2SpectrumReportersOption;
    private javax.swing.JRadioButton ms3SpectrumReportersOption;
    private javax.swing.JTextField mzTolTxt;
    private javax.swing.JLabel mzToleranceLabel;
    private javax.swing.JButton okButton;
    private javax.swing.JComboBox ppmCmb;
    private javax.swing.JRadioButton precursorMatchingReportersOption;
    private javax.swing.JLabel reporterIonMzToleranceLabel;
    private javax.swing.ButtonGroup reporterLocationButtonGroup;
    private javax.swing.JPanel reporterLocationPanel;
    private javax.swing.JTextField rtTolTxt;
    private javax.swing.JLabel rtToleranceLabel;
    private javax.swing.JPanel spectrumAnalysisPanel;
    // End of variables declaration//GEN-END:variables

    /**
     * Indicates whether the user canceled the editing.
     *
     * @return a boolean indicating whether the user canceled the editing
     */
    public boolean isCanceled() {
        return canceled;
    }

    /**
     * Updates the reporter location options.
     */
    private void updateReporterLocationOptions() {

        // enable or disable the precursor matching options
        mzToleranceLabel.setEnabled(precursorMatchingReportersOption.isSelected());
        mzTolTxt.setEnabled(precursorMatchingReportersOption.isSelected());
        ppmCmb.setEnabled(precursorMatchingReportersOption.isSelected());
        rtToleranceLabel.setEnabled(precursorMatchingReportersOption.isSelected());
        rtTolTxt.setEnabled(precursorMatchingReportersOption.isSelected());

    }

    /**
     * Methods which validates the user input (returns false in case of wrong
     * input).
     *
     * @return true if the input can be processed
     */
    private boolean validateInput() {

        // check the ion torerance
        Double input;

        try {
            input = Double.valueOf(ionToleranceTxt.getText().trim());
        } catch (Exception e) {

            JOptionPane.showMessageDialog(
                    this,
                    "Please input a number for the ion tolerance.",
                    "Ion Tolerance Error",
                    JOptionPane.ERROR_MESSAGE
            );

            return false;

        }

        if (input <= 0) {

            JOptionPane.showMessageDialog(
                    this,
                    "Please input a positive number for the ion tolerance.",
                    "Ion Tolerance Error",
                    JOptionPane.ERROR_MESSAGE
            );

            return false;
        }

        if (reporterMethod != null) {

            for (String reagent1 : reporterMethod.getReagentNames()) {

                for (String reagent2 : reporterMethod.getReagentNames()) {

                    if (!reagent1.equals(reagent2) && Math.abs(reporterMethod.getReagent(reagent1).getReporterIon().getTheoreticMass()
                            - reporterMethod.getReagent(reagent2).getReporterIon().getTheoreticMass()) <= input) {

                        JOptionPane.showMessageDialog(
                                this,
                                "The selected ion tolerance does not make it possible to distinguish " + reagent1 + " and " + reagent2 + ".",
                                "Ion Tolerance Error",
                                JOptionPane.WARNING_MESSAGE
                        );

                        return false;

                    }
                }

            }

        }

        // check the precursor matching
        if (precursorMatchingReportersOption.isSelected()) {

            try {
                input = Double.valueOf(mzTolTxt.getText().trim());
            } catch (Exception e) {

                JOptionPane.showMessageDialog(
                        this,
                        "Please input a number for the precursor m/z tolerance.",
                        "Matching Tolerance Error",
                        JOptionPane.ERROR_MESSAGE
                );

                return false;

            }

            if (input <= 0) {

                JOptionPane.showMessageDialog(
                        this,
                        "Please input a positive number for the precursor m/z tolerance.",
                        "Matching Tolerance Error",
                        JOptionPane.ERROR_MESSAGE
                );

                return false;

            }
            try {

                input = Double.valueOf(rtTolTxt.getText().trim());

            } catch (Exception e) {

                JOptionPane.showMessageDialog(
                        this,
                        "Please input a number for the precursor RT tolerance.",
                        "Matching Tolerance Error",
                        JOptionPane.ERROR_MESSAGE
                );

                return false;
            }

            if (input <= 0) {

                JOptionPane.showMessageDialog(
                        this,
                        "Please input a positive number for the precursor RT tolerance.",
                        "Matching Tolerance Error",
                        JOptionPane.ERROR_MESSAGE
                );

                return false;

            }

        }

        return true;

    }

    /**
     * Returns the settings as set by the user.
     *
     * @return the settings as set by the user
     */
    public ReporterIonSelectionSettings getReporterIonSelectionSettings() {

        ReporterIonSelectionSettings reporterIonSelectionSettings = new ReporterIonSelectionSettings();
        Double ionTolerance = Double.valueOf(ionToleranceTxt.getText().trim());
        reporterIonSelectionSettings.setReporterIonsMzTolerance(ionTolerance);
        reporterIonSelectionSettings.setMostAccurate(ionSelectionComboBox.getSelectedIndex() == 0);

        if (ms2SpectrumReportersOption.isSelected()) {

            reporterIonSelectionSettings.setReporterIonsLocation(ReporterIonsLocationType.ms2Spectra);

        } else if (ms3SpectrumReportersOption.isSelected()) {

            reporterIonSelectionSettings.setReporterIonsLocation(ReporterIonsLocationType.ms3Spectra);

        } else if (precursorMatchingReportersOption.isSelected()) {

            Double matchingMzTolerance = Double.valueOf(mzTolTxt.getText().trim());
            reporterIonSelectionSettings.setPrecursorMzTolerance(matchingMzTolerance);
            Double matchingRtTolerance = Double.valueOf(rtTolTxt.getText().trim());
            reporterIonSelectionSettings.setPrecursorMzTolerance(matchingRtTolerance);
            reporterIonSelectionSettings.setPrecursorMzPpm(ppmCmb.getSelectedIndex() == 0);

            reporterIonSelectionSettings.setReporterIonsLocation(ReporterIonsLocationType.precursorMatching);

        }

        return reporterIonSelectionSettings;

    }
}
