/*
 * Decompiled with CFR 0.152.
 */
package edu.ucsd.msjava.suffixarray;

import edu.ucsd.msjava.msutil.AminoAcid;
import edu.ucsd.msjava.msutil.Peptide;
import edu.ucsd.msjava.suffixarray.MatchSet;
import edu.ucsd.msjava.suffixarray.SuffixArraySequence;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.TreeSet;

public class MassArray {
    private static final String MASS_FILE_EXTENSION = ".marray";
    private static final float MAX_GAP_MASS = 1000.0f;
    private static final int INT_BYTE_SIZE = 4;
    private static final int FLOAT_BYTE_SIZE = 4;
    private int size;
    private float[] masses;
    private IntBuffer starts;
    private IntBuffer ends;

    public MassArray(SuffixArraySequence sequence, String massArrayFile) {
        int id;
        if (!new File(massArrayFile).exists()) {
            this.createMassArrayFile(sequence, massArrayFile);
        }
        if ((id = this.readMassArrayFile(massArrayFile)) != sequence.getId()) {
            System.err.println(massArrayFile + " was not created from the sequence " + sequence.getBaseFilepath());
            System.err.println("Please recreate the suffix array file by deleting the .canno, .cseq, and .csarr files.");
            System.exit(-1);
        }
    }

    public MassArray(SuffixArraySequence sequence) {
        this(sequence, sequence.getBaseFilepath() + MASS_FILE_EXTENSION);
    }

    public MatchSet findAll(float lower, float upper) {
        int matchIndex = Arrays.binarySearch(this.masses, lower);
        if (matchIndex < 0) {
            matchIndex = -(matchIndex + 1);
        }
        for (int i = Math.min(matchIndex, this.size - 1); i >= 0; --i) {
            if (!(this.masses[i] < lower)) continue;
            matchIndex = i + 1;
            break;
        }
        MatchSet ms = new MatchSet();
        for (int i = matchIndex; i < this.size && !(this.masses[i] > upper); ++i) {
            ms.add(this.starts.get(i), this.ends.get(i));
        }
        return ms;
    }

    private static void debug() {
        long time = System.currentTimeMillis();
        String userHome = System.getProperty("user.home");
        String fastaFile = userHome + "/Data/Databases/small.fasta";
        System.out.println("File name: " + fastaFile);
        SuffixArraySequence fs = new SuffixArraySequence(fastaFile);
        System.out.println("Total number of characters: " + fs.getSize());
        System.out.println("Alphabet size: " + fs.getAlphabetSize());
        System.out.println("Time to complete: " + (double)(System.currentTimeMillis() - time) / 1000.0 + "s");
        MassArray ma = new MassArray(fs);
        MatchSet ms = ma.findAll(500.0f, 500.1f);
        for (int i = 0; i < ms.getSize(); ++i) {
            int start = ms.getStart(i);
            int end = ms.getEnd(i);
            String match = fs.getSubsequence(start, end);
            System.out.println("Start: " + start + ". End: " + end + ". Sequence: " + match + " .Mass: " + Peptide.getMassFromString(match));
        }
    }

    public static void main(String[] args) {
        MassArray.debug();
    }

    private int readMassArrayFile(String massArrayFile) {
        try {
            DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(massArrayFile)));
            this.size = in.readInt();
            int id = in.readInt();
            FileChannel fc = new FileInputStream(massArrayFile).getChannel();
            int startPos = 8;
            int sizeOfMasses = this.size * 4;
            FloatBuffer fb = fc.map(FileChannel.MapMode.READ_ONLY, startPos, sizeOfMasses).asFloatBuffer();
            if (fb.hasArray()) {
                this.masses = fb.array();
            } else {
                this.masses = new float[this.size];
                for (int i = 0; i < this.size; ++i) {
                    this.masses[i] = fb.get(i);
                }
            }
            int startsSize = this.size * 4;
            this.starts = fc.map(FileChannel.MapMode.READ_ONLY, startPos += sizeOfMasses, startsSize).asIntBuffer();
            int endsSize = this.size * 4;
            this.ends = fc.map(FileChannel.MapMode.READ_ONLY, startPos += startsSize, endsSize).asIntBuffer();
            fc.close();
            return id;
        }
        catch (IOException e) {
            e.printStackTrace();
            System.exit(-1);
            return 0;
        }
    }

    private void createMassArrayFile(SuffixArraySequence sequence, String massArrayFile) {
        System.out.println("Creating the mass array sorted file of size: " + sequence.getSize() + " characters");
        TreeSet<MassObject> sortedMasses = new TreeSet<MassObject>();
        int sequenceLen = (int)sequence.getSize();
        for (int index = 0; index < sequenceLen; ++index) {
            AminoAcid aa;
            float cumMass = 0.0f;
            int newIndex = index;
            while (cumMass < 1000.0f && newIndex < sequenceLen && (aa = AminoAcid.getStandardAminoAcid(sequence.getCharAt(newIndex))) != null) {
                sortedMasses.add(new MassObject(cumMass += aa.getMass(), index, ++newIndex));
            }
        }
        System.out.println("Length of mass array " + sortedMasses.size());
        try {
            DataOutputStream out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(massArrayFile)));
            out.writeInt(sortedMasses.size());
            out.writeInt(sequence.getId());
            for (MassObject mo : sortedMasses) {
                out.writeFloat(mo.mass);
            }
            for (MassObject mo : sortedMasses) {
                out.writeInt(mo.start);
            }
            for (MassObject mo : sortedMasses) {
                out.writeInt(mo.end);
            }
            out.close();
        }
        catch (IOException e) {
            e.printStackTrace();
            System.exit(-1);
        }
    }

    private static class MassObject
    implements Comparable<MassObject> {
        float mass;
        int start;
        int end;

        public MassObject(float mass, int start, int end) {
            this.mass = mass;
            this.start = start;
            this.end = end;
        }

        @Override
        public int compareTo(MassObject other) {
            if (this.mass > other.mass) {
                return 1;
            }
            if (this.mass < other.mass) {
                return -1;
            }
            if (this.start > other.start) {
                return 1;
            }
            if (this.start < other.start) {
                return -1;
            }
            if (this.end > other.end) {
                return 1;
            }
            if (this.end < other.end) {
                return -1;
            }
            return 0;
        }
    }
}

